﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using Advent.aiDevices;

namespace Advent.Controls
{
    public partial class AI7280DCRecord : UserControl
    {

        AI7280_CO_Simulator.DCRecordingManager Recorder = null;  

        public AI7280DCRecord()
        {
            InitializeComponent();

            cbo_Mode.Items.Clear();
            cbo_Mode.Items.Add(RecordingMode.Indefinitely);
            cbo_Mode.Items.Add(RecordingMode.Single_Shot);
            cbo_Mode.SelectedIndex = 0;

            this.EnabledChanged += new EventHandler(AI7280DCRecordCtrl_EnabledChanged);
        }

        enum RecordingMode
        {
            Indefinitely,
            Single_Shot
        }


        void AI7280DCRecordCtrl_EnabledChanged(object sender, EventArgs e)
        {
            timer_Update.Enabled = this.Enabled;
        }

        public void SetRecorder(AI7280_CO_Simulator.DCRecordingManager Recorder)
        {
            this.Recorder = Recorder;
            if (Recorder ==null) return;
            

            // Add each sample rate ot the list
            cbo_SampleRates.Items.Clear();
            foreach (Frequency srate in Recorder.SupportedSampleRates)
            {
                cbo_SampleRates.Items.Add(srate);
            }
            cbo_SampleRates.SelectedIndex = 0;


            txt_SampleCount.Text = Recorder.RecordingDepth().ToString();

        }

        /// <summary>
        /// This is called asynchronously (from another thread) when the download completes
        /// </summary>
        /// <param name="writer"></param>
        private void DCDownloadComplete(SampleWriter writer)
        {

            WaveSampleWriter wr = writer as WaveSampleWriter;

            MessageBox.Show("DC Download to " + wr.FileName + " complete!");
        }

        private void timer_Update_Tick(object sender, EventArgs e)
        {
            try
            {
                if (Recorder == null) return;

                String Status = "";

                if (Recorder.IsRecording)
                {
                    if (Recorder.IsRecordingIndefinitely)
                        Status = "(Recording indefinitely)";
                    else
                        Status = "(Recording)";
                }
                else
                {
                    Status = "(Idle)";
                }
                Status += ("  [Recorded " + Recorder.RecordedTime.ToString() + "]");

                if (Recorder.IsDownloading) Status += " [Downloading]";

                lbl_Status.Text = Status;

            }
            catch { }
        }

        private void btn_Record_Click(object sender, EventArgs e)
        {
            try
            {
                // Select the sample rate
                Frequency SampleRate = cbo_SampleRates.SelectedItem as Frequency;
                if (SampleRate == null) throw new Exception("Invalid Sample rate selection");



                //Determine the number of samples specified
                int SampleCount = 0;
                if (!int.TryParse(txt_SampleCount.Text, out SampleCount)) throw new Exception("Invalid number of samples specified");

                switch ((RecordingMode)cbo_Mode.SelectedItem)
                {
                    case RecordingMode.Indefinitely:
                        /* This will start recording indefinitely until stopped. When stopped the maximum number of samples
                        * able to be recalled will be SampleCount.
                        * */
                        Recorder.StartRecordingIndefinitely();
                        break;

                    case RecordingMode.Single_Shot:
                        // This starts recording samples for exactly SampleCount samples, after which the AI-7280 will stop recording.
                        Recorder.StartRecording(SampleCount);
                        break;
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void btn_DownloadAC_Click(object sender, EventArgs e)
        {
            try
            {
                // we prompt the user for a wave file to save the output
                using (System.Windows.Forms.SaveFileDialog f = new SaveFileDialog())
                {
                    f.FileName = "AI7280 DC Recording.wav";
                    f.Filter = "Wave file (*.wav)|*.wav";
                    f.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.Desktop);

                    if (f.ShowDialog() != DialogResult.OK) return;

                    /*This starts downloading a wave file to the file specified through the sample writer interface
                     * Note: 
                     *  1)You can poll the .IsFinished property of a WaveSampleWriter to determine when it has completed
                     *  2) You can call the .Close function of the WaveSampleWriter to terminate the download at any time
                     * */
                    Recorder.StartDownload(new WaveSampleWriter(f.FileName), DCDownloadComplete);

                }

            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }   
        }

        private void btn_Stop_Click(object sender, EventArgs e)
        {
            try
            {
                Recorder.StopRecording();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }





    
    
    }
}
